/**
* \file: CarPlayAudioCodec.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPLay
*
* \author: Sudha Kuppusamy/ sudha.kuppusamy@in.bosch.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

// This file is a modified copy of APSAudioConverterStub.c
/*
    Copyright (C) 2013 Apple Inc. All Rights Reserved. Not to be used or disclosed without permission from Apple.
*/

#include "CarPlayAudioCodec.h"

using namespace std;

LOG_IMPORT_CONTEXT(cply_codec);

static std::map<std::pair<uint32_t, uint32_t>, AudioCodec> codecMap = {
  { { kAudioFormatMPEG4AAC, kAudioFormatLinearPCM }, AAC_LC_Decoder},
  { { kAudioFormatOpus, kAudioFormatLinearPCM }, OPUS_Decoder},
  { { kAudioFormatLinearPCM, kAudioFormatOpus }, OPUS_Encoder},
};

//===========================================================================================================================
//	AudioConverterNew
//===========================================================================================================================

OSStatus AudioConverterNew( const AudioStreamBasicDescription*	inSourceFormat, const AudioStreamBasicDescription*	inDestinationFormat, AudioConverterRef* outAudioConverter )
{
	OSStatus						err = kNoErr;
	AudioConverterPrivateRef		me;

	// Sample rate conversion and mixing are not supported
	if(( inDestinationFormat->mSampleRate != inSourceFormat->mSampleRate ) || ( inDestinationFormat->mChannelsPerFrame != inSourceFormat->mChannelsPerFrame ))
		return kUnsupportedErr;

	me = (AudioConverterPrivateRef) calloc( 1, sizeof( *me ) );

	if( me != NULL)
    {
        // Parameters are provided here to initialize the codec
        me->bitsPerChannel  = inDestinationFormat->mBitsPerChannel;
        me->sourceFormatID  = inSourceFormat->mFormatID;
        me->destFormatID    = inDestinationFormat->mFormatID;
        me->sampleRate      = inDestinationFormat->mSampleRate;
        me->channels        = inDestinationFormat->mChannelsPerFrame;
        me->framesPerPacket = inDestinationFormat->mFramesPerPacket;

        map<std::pair<uint32_t, uint32_t>, AudioCodec>::iterator it;
        it = codecMap.find(std::make_pair(me->sourceFormatID, me->destFormatID));

        if (it != codecMap.end())
        {
            err = audioConverterCreateGstPipeline(me, it->second);
        }
        else
        {
            LOG_ERROR((cply_codec, "CodecError: Unsupported source or destination format"));
            err = kUnsupportedErr;
        }

        if(err == kNoErr)
        {
            *outAudioConverter = (AudioConverterRef) me;
             me = NULL;
        }
        else
        {
            AudioConverterDispose( (AudioConverterRef) me );
        }
    }
	else
	{
	    err = kNoMemoryErr;
	}

	return( err );
}

//===========================================================================================================================
//	AudioConverterDispose
//===========================================================================================================================

OSStatus AudioConverterDispose( AudioConverterRef inConverter )
{
	AudioConverterPrivateRef const me = (AudioConverterPrivateRef) inConverter;

	if( me->nativeCodecRef )
	{
	    audioConverterDestroyGstPipeline(me);
	}

	free( me );

	return( kNoErr );
}

//===========================================================================================================================
//	AudioConverterReset
//===========================================================================================================================

OSStatus AudioConverterReset( AudioConverterRef inConverter )
{
    OSStatus err = kNoErr;

    AudioConverterPrivateRef const      me = (AudioConverterPrivateRef) inConverter;

    LOGD_VERBOSE((cply_codec, "CodecError: flushing audio codec pipeline"));
#if 0  // commenting out since there is some issue in sending flush event in gstreamer0.10
    flush_start = gst_event_new_flush_start();
    err = audioConverterSendGstEvent(me->nativeCodecRef, flush_start);

    if (err == kNoErr)
    {
#if GST_CHECK_VERSION(1,0,0)
        flush_stop = gst_event_new_flush_stop(TRUE);
#else
        flush_stop = gst_event_new_flush_stop();
#endif

        err = audioConverterSendGstEvent(me->nativeCodecRef, flush_stop);
        if(err == kNoErr)
        {
#if GST_CHECK_VERSION(1,0,0)
            GstSegment *gstSegment;
            gstSegment = gst_segment_new();
            gst_segment_init( gstSegment, GST_FORMAT_TIME );
            segment_event = gst_event_new_segment(gstSegment);
#else
            segment_event = gst_event_new_new_segment(TRUE, 1.0, GST_FORMAT_TIME, 0, -1, 0);
#endif
            err = audioConverterSendGstEvent(me->nativeCodecRef, segment_event);
        }
    }
#endif

    GstStateChangeReturn ret;
    OSStatus EosErr;

    EosErr = _AudioConverterSendEOS(me);
    if(EosErr < 0)
        LOG_ERROR((cply_codec, "CodecError: failed to send Eos event to appsrc"));
    else
    {
        // setting pipeline state to NULL will flush the buffer
        ret = gst_element_set_state (GST_ELEMENT(me->nativeCodecRef), GST_STATE_NULL);
        if(ret == GST_STATE_CHANGE_FAILURE)
        {
            LOG_ERROR((cply_codec, "CodecError: Error in setting the pipeline to NULL state"));
                        audioConverterDestroyGstPipeline(me);
            err = kUnknownErr;
        }
        // setting pipeline state to READY will flush the buffer
        ret = gst_element_set_state (GST_ELEMENT(me->nativeCodecRef), GST_STATE_READY);
        if(ret == GST_STATE_CHANGE_FAILURE)
        {
            LOG_ERROR((cply_codec, "CodecError: Error in setting the pipeline to READY state"));
                        audioConverterDestroyGstPipeline(me);
            err = kUnknownErr;
        }
        // setting pipeline state to PLAYING to continue playback
        ret = gst_element_set_state (GST_ELEMENT(me->nativeCodecRef), GST_STATE_PLAYING);
        if(ret == GST_STATE_CHANGE_FAILURE)
        {
            LOG_ERROR((cply_codec, "CodecError: Error in setting the pipeline to PLAYING state"));
            audioConverterDestroyGstPipeline(me);
            err = kUnknownErr;
        }
    }

    return err;
}

static OSStatus _AudioConverterSetPropertyAACDecode( AudioConverterPrivateRef const me, AudioConverterPropertyID inPropertyID, uint32_t inSize, const void* inData )
{
	(void)inSize;
	(void)inData;
	(void)me;

	switch ( inPropertyID )
	{
		default:
			return kUnsupportedErr;
	}
	return kNoErr;
}

static OSStatus _AudioConverterSetPropertyOpusDecode( AudioConverterPrivateRef const me, AudioConverterPropertyID	inPropertyID, uint32_t inSize, const void* inData )
{
	(void)me;
	(void)inSize;
	(void)inData;

	switch ( inPropertyID )
	{
		default:
			return kUnsupportedErr;
	}
}

static OSStatus _AudioConverterSetPropertyOpusEncode( AudioConverterPrivateRef const me, AudioConverterPropertyID inPropertyID, uint32_t inSize, const void* inData )
{
	(void)me;
	(void)inSize;
	(void)inData;

	switch ( inPropertyID )
	{
		case kAudioCodecPropertyPacketSizeLimitForVBR:
		{
			if( inSize != sizeof( uint32_t ) )
				return kSizeErr;

			// $$$ TODO: Set up encoder properties
			return kNoErr;
		}
		default:
			return kUnsupportedErr;
	}
}

//===========================================================================================================================
//	AudioConverterSetProperty
//===========================================================================================================================

OSStatus AudioConverterSetProperty( AudioConverterRef inConverter, AudioConverterPropertyID	inPropertyID, uint32_t inSize, const void* inData )
{
	AudioConverterPrivateRef const		me = (AudioConverterPrivateRef) inConverter;

	if( !me->nativeCodecRef )
		return kStateErr;
	switch( me->sourceFormatID )
	{
		case kAudioFormatMPEG4AAC:
			return _AudioConverterSetPropertyAACDecode( me, inPropertyID, inSize, inData );

		case kAudioFormatOpus:
			return _AudioConverterSetPropertyOpusDecode( me, inPropertyID, inSize, inData );

		case kAudioFormatLinearPCM:
			if( me->destFormatID == kAudioFormatOpus )
				return _AudioConverterSetPropertyOpusEncode( me, inPropertyID, inSize, inData );
			else
			    return kUnsupportedErr;

		default:
			return kUnsupportedErr;
	}
}

static OSStatus _AudioConverterFillComplexBufferAACDecode( AudioConverterRef inConverter, AudioConverterComplexInputDataProc inInputDataProc, void* inInputDataProcUserData, uint32_t* ioOutputDataPacketSize, AudioBufferList* outOutputData, AudioStreamPacketDescription* outPacketDescription )
{
	AudioConverterPrivateRef const me  = (AudioConverterPrivateRef) inConverter;
	OSStatus                       err = kNoErr;

	AudioBufferList                bufferList;
	uint32_t                       packetCount;
	AudioStreamPacketDescription*  packetDesc;

    if(outOutputData == NULL)
    {
        LOG_ERROR((cply_codec, "CodecError: Provided output bufferlist is empty"));
        return kUnknownErr;
    }
	// Callback will provide number of frames per packet in ioOutputDataPacketSize
	if( *ioOutputDataPacketSize < kAudioSamplesPerPacket_AAC_LC )
	{
		return kSizeErr;
	}

	// Request 1 packet of AAC LC through callback to decode kAudioSamplesPerPacket_AAC_LC of output
	packetCount = 1;
	packetDesc  = NULL;

	err = inInputDataProc( inConverter, &packetCount, &bufferList, &packetDesc, inInputDataProcUserData );
	if(err == kNoErr)
	{
        err = _AudioConverterConvertAudioData(me, bufferList.mBuffers[0].mData, bufferList.mBuffers[0].mDataByteSize, outOutputData);
        if( err == kNoErr )
        {
            if(outPacketDescription)
            {
                outPacketDescription[ 0 ].mStartOffset              = 0;
                outPacketDescription[ 0 ].mVariableFramesInPacket   = 0;
                outPacketDescription[ 0 ].mDataByteSize             = outOutputData->mBuffers[ 0 ].mDataByteSize;
            }
            // Set the number of samples produced
            *ioOutputDataPacketSize = kAudioSamplesPerPacket_AAC_LC;
        }
        else
        {
            LOG_ERROR((cply_codec, "CodecError: Error in decoding!!"));
            *ioOutputDataPacketSize = 0;
        }
	}
	else
    {
        LOG_ERROR((cply_codec, "CodecError: No audio samples received for decoding !!"));
    }

	return err;
}

static OSStatus _AudioConverterFillComplexBufferOpusDecode( AudioConverterRef inConverter, AudioConverterComplexInputDataProc inInputDataProc, void* inInputDataProcUserData, uint32_t* ioOutputDataPacketSize, AudioBufferList* outOutputData, AudioStreamPacketDescription* outPacketDescription )
{
	AudioConverterPrivateRef const me = (AudioConverterPrivateRef) inConverter;
	OSStatus							err;

	AudioBufferList						bufferList;
	uint32_t							packetCount;
	AudioStreamPacketDescription*		packetDesc;

	(void)outPacketDescription;
	(void)ioOutputDataPacketSize;
	(void)outOutputData;
	(void)me;

	// $$$ TODO: Callback will provide number of frames per packet in ioOutputDataPacketSize
	bufferList.mNumberBuffers = 1;
	packetCount = 1;
	packetDesc  = NULL;

	// $$$ TODO: Request 1 packet of OPUS through callback to decode 20 ms of output.
	// The codec is responsible for consuming all bytes provided.
	err = inInputDataProc( inConverter, &packetCount, &bufferList, &packetDesc, inInputDataProcUserData );
	if(err == kNoErr)
	{
        err = _AudioConverterConvertAudioData(me, bufferList.mBuffers[0].mData, bufferList.mBuffers[0].mDataByteSize, outOutputData);
        if( err == kNoErr )
        {
            if(outPacketDescription)
            {
                outPacketDescription[ 0 ].mStartOffset              = 0;
                outPacketDescription[ 0 ].mVariableFramesInPacket   = 0;
                outPacketDescription[ 0 ].mDataByteSize             = outOutputData->mBuffers[ 0 ].mDataByteSize;
            }
        }
        else
        {
            LOG_ERROR((cply_codec, "CodecError: Error in decoding!!"));
        }
    }
    else
    {
        LOG_ERROR((cply_codec, "CodecError: No audio samples received for decoding !!"));
}

	return err;
}

static OSStatus _AudioConverterFillComplexBufferOpusEncode( AudioConverterRef inConverter, AudioConverterComplexInputDataProc inInputDataProc, void* inInputDataProcUserData, uint32_t* ioOutputDataPacketSize, AudioBufferList* outOutputData, AudioStreamPacketDescription* outPacketDescription )
{
    AudioConverterPrivateRef const me = (AudioConverterPrivateRef) inConverter;
    OSStatus                       err = kNoErr;
    uint32_t                       numPacket = 0;
    int32_t                        encoded_len = 0;
    uint32_t                       pcm_length = 0;

    AudioBufferList                bufferList;
    uint32_t                       packetCount;
    AudioStreamPacketDescription*  packetDesc;
    uint8_t*                       tempBuffer;

    // Encode 20ms worth of samples per output packet
    bufferList.mNumberBuffers = 1;
    packetCount               = me->framesPerPacket;
    packetDesc                = NULL;
    pcm_length                = me->framesPerPacket*me->channels*sizeof(int16_t);

    // Request number of frames (packetCount) to produce 1 packet representing 20 ms. The number
    // of frames provided will be returned in packetCount.  If the number of frames received is less  than
    // the requested amount, save the data internally and return no packets encoded in ioOutputDataPacketSize.
    err = inInputDataProc( inConverter, &packetCount, &bufferList, &packetDesc, inInputDataProcUserData );
    if(err == kUnderrunErr)
    {
        LOG_ERROR((cply_codec, "CodecError: UnderRun Error in OPUS Encoding"));
        return err;
    }

    tempBuffer = static_cast<uint8_t*>(bufferList.mBuffers[ 0 ].mData);

    // Buffering
    //check whether local buffer can hold the received samples
    if(pcm_length - me->pcmBufferPos >= bufferList.mBuffers[ 0 ].mDataByteSize)
    {
        memcpy(&me->pcmBuffer[me->pcmBufferPos], tempBuffer, bufferList.mBuffers[ 0 ].mDataByteSize);

        me->pcmBufferPos += bufferList.mBuffers[ 0 ].mDataByteSize;

        if(me->pcmBufferPos == pcm_length)
        {
            // encode
            err = _AudioConverterConvertAudioData(me, me->pcmBuffer, me->pcmBufferPos, outOutputData);

            encoded_len = outOutputData->mBuffers[ 0 ].mDataByteSize;
            me->pcmBufferPos = 0;
        }
    }
    else
    {
        //copy only part of the buffer, encode and then store remaining samples
        uint32_t SamplesToCopy = pcm_length - me->pcmBufferPos;

        memcpy(&me->pcmBuffer[me->pcmBufferPos], tempBuffer, SamplesToCopy);
        me->pcmBufferPos += SamplesToCopy;

        // encode
        err = _AudioConverterConvertAudioData(me, me->pcmBuffer, me->pcmBufferPos, outOutputData);

        encoded_len = outOutputData->mBuffers[ 0 ].mDataByteSize;
        me->pcmBufferPos = 0;

        memcpy(&me->pcmBuffer[me->pcmBufferPos], &tempBuffer[SamplesToCopy], bufferList.mBuffers[ 0 ].mDataByteSize - SamplesToCopy);

        me->pcmBufferPos = bufferList.mBuffers[ 0 ].mDataByteSize-SamplesToCopy;
    }

    // Fill number of encoded packets
    if(err == kNoErr)
    {
        if(encoded_len > 0)
        {
            if( outPacketDescription )
            {
                //Fill out outputPacketDescription with the output results
                outPacketDescription[ numPacket ].mStartOffset = 0;
                outPacketDescription[ numPacket ].mVariableFramesInPacket = 0;
                outPacketDescription[ numPacket ].mDataByteSize = outOutputData->mBuffers[ 0 ].mDataByteSize;
            }

            // Set the number of packets encoded
            *ioOutputDataPacketSize = 1;
        }
        else
        {
            // samples saved locally. So set no. of packets encoded as 0 and return kNoErr
            *ioOutputDataPacketSize = 0;
        }
    }
    else
    {
        // failure in encoding
        LOG_ERROR((cply_codec, "CodecError: error in encoding"));
    }

    return err;
}

//===========================================================================================================================
//	AudioConverterFillComplexBuffer
//===========================================================================================================================

OSStatus AudioConverterFillComplexBuffer( AudioConverterRef inConverter, AudioConverterComplexInputDataProc inInputDataProc,
		void*								inInputDataProcUserData,
		uint32_t*							ioOutputDataPacketSize,
		AudioBufferList*					outOutputData,
		AudioStreamPacketDescription*		outPacketDescription )
{
    OSStatus err = kNoErr;
	AudioConverterPrivateRef const me = (AudioConverterPrivateRef) inConverter;

	if( !me || !me->pipelineCreated )
	    return kStateErr;

	switch ( me->sourceFormatID )
	{
		case kAudioFormatMPEG4AAC:
			// AAC LC to PCM
			err =  _AudioConverterFillComplexBufferAACDecode(  inConverter, inInputDataProc, inInputDataProcUserData, ioOutputDataPacketSize, outOutputData, outPacketDescription );
			break;

		case kAudioFormatOpus:
			// Opus to PCM
			err = _AudioConverterFillComplexBufferOpusDecode( inConverter, inInputDataProc, inInputDataProcUserData, ioOutputDataPacketSize, outOutputData, outPacketDescription );
			break;

		case kAudioFormatLinearPCM:
            if( me->destFormatID == kAudioFormatOpus )
            {
                // PCM to Opus
                err = _AudioConverterFillComplexBufferOpusEncode( inConverter, inInputDataProc, inInputDataProcUserData, ioOutputDataPacketSize, outOutputData, outPacketDescription );
            }
            else
            {
                err = kUnsupportedErr;
            }
            break;
		default:
		    err = kUnsupportedErr;
	}

	return err;
}
